/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 1999-2001 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */


#ifndef _NV_H_
#define _NV_H_

#include <nvtypes.h>

/* Nvidia's reserved major device number
 * Set this to 0 to request dynamic major number allocation. */
#define NV_MAJOR_DEVICE_NUMBER 195

/* most cards in a single system */
#define NV_MAX_DEVICES 8

#define NV_IOCTL_MAGIC      'F'
#define NV_IOCTL_BASE       200
#define NV_ESC_CARD_INFO         (NV_IOCTL_BASE + 0)
#define NV_ESC_ENV_INFO          (NV_IOCTL_BASE + 2)
#define NV_ESC_RM_API_VERSION    (NV_IOCTL_BASE + 3)
#define NV_ESC_MODULE_RESET      (NV_IOCTL_BASE + 4)
#define NV_ESC_ALLOC_OS_EVENT    (NV_IOCTL_BASE + 6)
#define NV_ESC_FREE_OS_EVENT     (NV_IOCTL_BASE + 7)
#define NV_ESC_POST_VBIOS        (NV_IOCTL_BASE + 8)
#define NV_ESC_STATUS_CODE       (NV_IOCTL_BASE + 9)
/* NOTE: using an ioctl number > 55 will overflow! */

/*
 * Solaris provides no more than 8 bits for the argument size in
 * the ioctl() command encoding; make sure we don't exceed this
 * limit.
 */
#if !defined(NV_SUNPRO_C)
#define __NV_IOWR_ASSERT(type) ((sizeof(type) <= 0xff) ? 1 : -1)
#define __NV_IOWR(nr, type) ({                                        \
    typedef char __NV_IOWR_TYPE_SIZE_ASSERT[__NV_IOWR_ASSERT(type)];  \
    _IOWR(NV_IOCTL_MAGIC, (nr), type);                                \
})
#else
#define __NV_IOWR(nr, type) (_IOWR(NV_IOCTL_MAGIC, (nr), type))
#endif

typedef struct nv_ioctl_card_info
{
    NvU16    flags;               /* see below                   */
    NvU16    bus;                 /* bus number (PCI, AGP, etc)  */
    NvU16    slot;                /* card slot                   */
    NvU16    vendor_id;           /* PCI vendor id               */
    NvU16    device_id;
    NvU16    interrupt_line;
    NvU32    reg_address;         /* register aperture           */
    NvU32    reg_size;
    NvU32    fb_address;          /* framebuffer aperture        */
    NvU32    fb_size;
} nv_ioctl_card_info_t;

#define NV_IOCTL_CARD_INFO_FLAG_PRESENT       0x0001
#define NV_IOCTL_CARD_INFO_FLAG_NEED_MSYNC    0x0002

#define NV_IOCTL_CARD_INFO     \
    __NV_IOWR(NV_ESC_CARD_INFO, NvU8[NV_MAX_DEVICES*sizeof(nv_ioctl_card_info_t)])


typedef struct nv_ioctl_post_vbios
{
    NvU32 bus;
    NvU32 slot;
} nv_ioctl_post_vbios_t;

/* notify resman which card is primary */
#define NV_IOCTL_POST_VBIOS __NV_IOWR(NV_ESC_POST_VBIOS, nv_ioctl_post_vbios_t)

typedef struct nv_ioctl_env_info
{
    NvU32 sim_env;
    NvU32 pat_supported;
} nv_ioctl_env_info_t;

#define NV_IOCTL_ENV_INFO  __NV_IOWR(NV_ESC_ENV_INFO, nv_ioctl_env_info_t)

/* rm api check
 * used to verify client/rm interaction both ways.
 * structure overloads card_info structure passed into resman, so resman can 
 * verify the client is using the correct version of the api.
 * client then does api_version ioctl to verify the rm is the correct version.
 * we do both to avoid cases where one but not the other supports api checking.
 * esp w/ recent memory/mmap changes to resman, we've seen some cases where 
 * mix-matching rm/clients can case serious damage.
 */
typedef struct nv_ioctl_rm_api_version
{
    NvU32 magic;
    NvU32 major;
    NvU32 minor;
    NvU32 patch;
} nv_ioctl_rm_api_version_t;

#define NV_RM_API_VERSION_MAGIC_REQ  0x0197fade
#define NV_RM_API_VERSION_MAGIC_REP  0xbead2929
#define NV_RM_API_VERSION_MAGIC_LAX_REQ         (NV_RM_API_VERSION_MAGIC_REQ ^ '1')
#define NV_RM_API_VERSION_MAGIC_OVERRIDE_REQ    (NV_RM_API_VERSION_MAGIC_REQ ^ '2')

#define NV_IOCTL_RM_API_VERSION __NV_IOWR(NV_ESC_RM_API_VERSION, nv_ioctl_rm_api_version_t)

/* debug tool to force module count to 0 */
#define NV_IOCTL_MODULE_RESET  _IO(NV_IOCTL_MAGIC, NV_ESC_MODULE_RESET)

/* alloc event */
typedef struct nv_ioctl_alloc_os_event
{
    NvU32 hClient;
    NvU32 hDevice;
    NvU32 hOsEvent;
    NvU32 fd;
    NvU32 Status;
} nv_ioctl_alloc_os_event_t;

#define NV_IOCTL_ALLOC_OS_EVENT __NV_IOWR(NV_ESC_ALLOC_OS_EVENT, nv_ioctl_alloc_os_event_t)

/* free event */
typedef struct nv_ioctl_free_os_event
{
    NvU32 hClient;
    NvU32 hDevice;
    NvU32 fd;
    NvU32 Status;
} nv_ioctl_free_os_event_t;

#define NV_IOCTL_FREE_OS_EVENT __NV_IOWR(NV_ESC_FREE_OS_EVENT, nv_ioctl_free_os_event_t)

/* status code */
typedef struct nv_ioctl_status_code
{
    NvU32 bus;
    NvU32 slot;
    NvU32 status;
} nv_ioctl_status_code_t;

#define NV_IOCTL_STATUS_CODE __NV_IOWR(NV_ESC_STATUS_CODE, nv_ioctl_status_code_t)

#ifdef __KERNEL__

/*
 * ptr arithmetic convenience
 */

typedef union
{
    volatile V008 Reg008[1];
    volatile V016 Reg016[1];
    volatile V032 Reg032[1];
} nv_hwreg_t, * nv_phwreg_t;


#define NVRM_PCICFG_NUM_BARS            6
#define NVRM_PCICFG_BAR_OFFSET(i)       (0x10 + (i) * 4)
#define NVRM_PCICFG_BAR_REQTYPE_MASK    0x00000001
#define NVRM_PCICFG_BAR_REQTYPE_MEMORY  0x00000000
#define NVRM_PCICFG_BAR_MEMTYPE_MASK    0x00000006
#define NVRM_PCICFG_BAR_MEMTYPE_64BIT   0x00000004
#define NVRM_PCICFG_BAR_ADDR_MASK       0xfffffff0

#define NV_GPU_NUM_BARS                 3
#define NV_GPU_BAR_INDEX_REGS           0
#define NV_GPU_BAR_INDEX_FB             1
#define NV_GPU_BAR_INDEX_IMEM           2

typedef struct {
    NvU32 address;
    NvU32 size;
    NvU32 offset;
    NvU32 *map;
    nv_phwreg_t map_u;
} nv_aperture_t;


typedef struct {
    char *node;
    char *name;
    U032 *data;
    BOOL  valid;
} nv_parm_t;

/*
 * this is a wrapper for unix events
 * unlike the events that will be returned to clients, this includes
 * kernel-specific data, such as file pointer, etc..
 */
typedef struct nv_event_s
{
    NvU32 hParent;
    NvU32 hObject;
    NvU32 index;
    void  *file;  /* per file-descriptor data pointer */
    NvU32 handle;
    NvU32 fd;
    struct nv_event_s *next;
} nv_event_t;

typedef struct nv_kern_mapping_s
{
    void  *addr;
    NvU32 size;
    struct nv_kern_mapping_s *next;
} nv_kern_mapping_t;


/*
 * per device state
 */

typedef struct
{
    void  *priv;                    /* private data */
    void  *os_state;                /* os-specific device state */

    int    rmInitialized;
    int    flags;

    /* PCI config info */
    U016 bus;
    U016 slot;
    U016 vendor_id;
    U016 device_id;

    /* physical characteristics */
    nv_aperture_t bars[NV_GPU_NUM_BARS];
    nv_aperture_t *regs;
    nv_aperture_t *fb;
    nv_aperture_t agp;

    NvU32  interrupt_line;

    U032 agp_config;
    U032 agp_status;

    /* save/restore eflags register */
    unsigned long saved_eflags;

    U032 sim_env;

    U032 rc_timer_enabled;

    /* list of events allocated for this device */
    nv_event_t *event_list;

    nv_kern_mapping_t *kern_mappings;

} nv_state_t;

/*
 * flags
 */

#define NV_FLAG_OPEN            0x0001
#define NV_FLAG_NEEDS_POSTING   0x0002
#define NV_FLAG_WAS_POSTED      0x0004
#define NV_FLAG_CONTROL         0x0010
#define NV_FLAG_HOTKEY_OCCURRED 0x0100
#define NV_FLAG_MAP_REGS_EARLY  0x0200

#define NV_PM_APM_SUSPEND       0x0001
#define NV_PM_APM_RESUME        0x0002
#define NV_PM_ACPI_STANDBY      0x0003
#define NV_PM_ACPI_RESUME       0x0004

#define NV_AGP_STATUS_DISABLED  0x0000
#define NV_AGP_STATUS_ENABLED   0x0001
#define NV_AGP_STATUS_FAILED    0x0002

#define NV_AGP_ENABLED(nv)      ((nv)->agp_status & NV_AGP_STATUS_ENABLED)
#define NV_AGP_FAILED(nv)       ((nv)->agp_status & NV_AGP_STATUS_FAILED)

#define NV_NVAGP_ENABLED(nv)    ((nv)->agp_config & NVOS_AGP_CONFIG_NVAGP)
#define NV_OSAGP_ENABLED(nv)    ((nv)->agp_config & NVOS_AGP_CONFIG_OSAGP)

/*
** where we hide our nv_state_t * ...
*/
#define NV_SET_NV_STATE(pgpu,p) ((pgpu)->pOsHwInfo = (p))
#define NV_GET_NV_STATE(pgpu)   ((nv_state_t *)(pgpu)->pOsHwInfo)

/* mmap(2) offsets */

#define IS_REG_OFFSET(nv, offset, length)                                      \
             (((offset) >= (nv)->regs->address) &&                             \
             (((offset) + ((length)-1)) <= (nv)->regs->address + ((nv)->regs->size-1)))

#define IS_FB_OFFSET(nv, offset, length)                                       \
             (((offset) >= (nv)->fb->address) &&                               \
             (((offset) + ((length)-1)) <= (nv)->fb->address + ((nv)->fb->size-1)))

#define IS_AGP_OFFSET(nv, offset, length)                                      \
             ((NV_AGP_ENABLED(nv)) &&                                          \
             ((offset) >= (nv)->agp.address) &&                                \
             (((offset) + ((length)-1)) <= (nv)->agp.address + ((nv)->agp.size-1)))

/* duplicated from nvos.h for external builds */
#ifndef NVOS_AGP_CONFIG_DISABLE_AGP
#  define NVOS_AGP_CONFIG_DISABLE_AGP (0x00000000)
#endif
#ifndef NVOS_AGP_CONFIG_NVAGP
#  define NVOS_AGP_CONFIG_NVAGP       (0x00000001)
#endif
#ifndef NVOS_AGP_CONFIG_OSAGP
#  define NVOS_AGP_CONFIG_OSAGP       (0x00000002)
#endif
#ifndef NVOS_AGP_CONFIG_ANYAGP
#  define NVOS_AGP_CONFIG_ANYAGP      (0x00000003)
#endif


/* device name length; must be atleast 8 */

#define NV_DEVICE_NAME_LENGTH 40


/*
 * driver internal interfaces
 */

/* need a fake device number for control device; just to flag it for msgs */
#define NV_CONTROL_DEVICE_NUMBER     100


#ifndef NVWATCH

/*
 * ---------------------------------------------------------------------------
 *
 * Function prototypes for UNIX specific OS interface.
 *
 * ---------------------------------------------------------------------------
 */

/*
 * Make sure that arguments to and from the core resource manager
 * are passed and expected on the stack. define duplicated in os-interface.h
 */
#if !defined(NV_API_CALL)
#if defined(NVCPU_X86)
#define NV_API_CALL __attribute__((regparm(0)))
#else
#define NV_API_CALL
#endif
#endif


void*  NV_API_CALL  nv_find_nv_mapping           (nv_state_t *, unsigned long); 
ULONG  NV_API_CALL  nv_find_dma_mapping          (nv_state_t *, unsigned long, void **, U032);
ULONG  NV_API_CALL  nv_dma_to_phys_address       (nv_state_t *, ULONG);
void*  NV_API_CALL  nv_find_agp_mapping          (nv_state_t *, unsigned long);
void*  NV_API_CALL  nv_find_agp_kernel_mapping   (nv_state_t *, unsigned long); 
ULONG  NV_API_CALL  nv_get_kern_phys_address     (ULONG);
ULONG  NV_API_CALL  nv_get_user_phys_address     (ULONG);
void*  NV_API_CALL  nv_get_adapter_state         (U016, U016);

void   NV_API_CALL  nv_lock_rm                   (nv_state_t *);
void   NV_API_CALL  nv_unlock_rm                 (nv_state_t *);

void   NV_API_CALL  nv_set_hotkey_occurred_flag  (void);
S032   NV_API_CALL  nv_int10h_call               (nv_state_t *, U032 *, U032 *, U032 *, U032 *, void *);

S032   NV_API_CALL  nv_alloc_pages               (nv_state_t *, void **, U032, U032, U032, U032, U032, U032, void **);
S032   NV_API_CALL  nv_free_pages                (nv_state_t *, void **, U032, U032, void *);

S032   NV_API_CALL  nv_agp_init                  (nv_state_t *, void **, void **, void *, U032);
S032   NV_API_CALL  nv_agp_teardown              (nv_state_t *);
S032   NV_API_CALL  nv_translate_address         (nv_state_t *, unsigned long, U032, unsigned long *);

S032   NV_API_CALL  nv_start_rc_timer            (nv_state_t *);
S032   NV_API_CALL  nv_stop_rc_timer             (nv_state_t *);

void   NV_API_CALL  nv_post_event                (nv_state_t *, nv_event_t *, U032, U032);
S032   NV_API_CALL  nv_get_event                 (nv_state_t *, void *, nv_event_t *, U032 *);

S032   NV_API_CALL  nv_no_incoherent_mappings    (void);

void   NV_API_CALL  nv_verify_pci_config         (nv_state_t *);


/*
 * ---------------------------------------------------------------------------
 *
 * Function prototypes for Resource Manager interface.
 *
 * ---------------------------------------------------------------------------
 */

BOOL       NV_API_CALL  rm_init_rm               (void);
BOOL       NV_API_CALL  rm_shutdown_rm           (void);
BOOL       NV_API_CALL  rm_init_private_state    (nv_state_t *);
BOOL       NV_API_CALL  rm_free_private_state    (nv_state_t *);
BOOL       NV_API_CALL  rm_init_adapter          (nv_state_t *);
BOOL       NV_API_CALL  rm_disable_adapter       (nv_state_t *);
U032       NV_API_CALL  rm_get_adapter_status    (nv_state_t *);
BOOL       NV_API_CALL  rm_shutdown_adapter      (nv_state_t *);
void       NV_API_CALL  rm_disable_interrupts    (void);
void       NV_API_CALL  rm_enable_interrupts     (void);
BOOL       NV_API_CALL  rm_ioctl                 (nv_state_t *, void *, U032, void *);
BOOL       NV_API_CALL  rm_isr                   (nv_state_t *, U032 *);
void       NV_API_CALL  rm_isr_bh                (nv_state_t *);
RM_STATUS  NV_API_CALL  rm_power_management      (nv_state_t *, U032, U032);
U032       NV_API_CALL  rm_get_vbios_version     (nv_state_t *, U032 *, U032 *, U032 *, U032 *, U032 *);
void       NV_API_CALL  rm_free_unused_clients   (nv_state_t *, U032, void *);
RM_STATUS  NV_API_CALL  rm_io_flush              (nv_state_t *);
U032       NV_API_CALL  rm_change_res_mode       (nv_state_t *, U032);
void       NV_API_CALL  rm_save_client           (nv_state_t *, U032, void *);
void       NV_API_CALL  rm_release_client        (nv_state_t *, U032);


void       NV_API_CALL  rm_update_agp_config     (nv_state_t *);
RM_STATUS  NV_API_CALL  rm_init_agp              (nv_state_t *);
RM_STATUS  NV_API_CALL  rm_teardown_agp          (nv_state_t *);

RM_STATUS  NV_API_CALL  rm_alloc_agp_pages       (nv_state_t *, void **, U032, U032, void **, U032 *);
RM_STATUS  NV_API_CALL  rm_free_agp_pages        (nv_state_t *, void **, void *);

RM_STATUS  NV_API_CALL  rm_alloc_agp_bitmap      (nv_state_t *, U032, U032 *);
RM_STATUS  NV_API_CALL  rm_free_agp_bitmap       (nv_state_t *, U032, U032);
RM_STATUS  NV_API_CALL  rm_set_agp_bitmap        (nv_state_t *, void *);
RM_STATUS  NV_API_CALL  rm_clear_agp_bitmap      (nv_state_t *, void **);

RM_STATUS  NV_API_CALL  rm_access_registry       (nv_state_t *, U032, U008 *, U032, U008 *, U032, U008 *, U032 *, U032 *, U032 *);
RM_STATUS  NV_API_CALL  rm_read_registry_dword   (nv_state_t *, U008 *, U008 *, U032 *);
RM_STATUS  NV_API_CALL  rm_write_registry_dword  (nv_state_t *, U008 *, U008 *, U032);
RM_STATUS  NV_API_CALL  rm_read_registry_binary  (nv_state_t *, U008 *, U008 *, U008 *, U032 *);
RM_STATUS  NV_API_CALL  rm_write_registry_binary (nv_state_t *, U008 *, U008 *, U008 *, U032);

RM_STATUS  NV_API_CALL  rm_run_rc_callback       (nv_state_t *);
RM_STATUS  NV_API_CALL  rm_get_device_name       (nv_state_t *, U032, U032, U008*);

RM_STATUS  NV_API_CALL  rm_alloc_os_event(nv_state_t *, U032, U032 *, void *, U032);
RM_STATUS  NV_API_CALL  rm_free_os_event(nv_state_t *, U032, U032);
RM_STATUS  NV_API_CALL  rm_get_event_data(nv_state_t *, void *, U008 *, U032 *);

RM_STATUS  NV_API_CALL  rm_validate_pfn_range    (void *, U032, U032);
U064       NV_API_CALL  nv_rdtsc                 (void);

void       NV_API_CALL  rm_register_ioctl_conversions(void);
void       NV_API_CALL  rm_unregister_ioctl_conversions(void);

BOOL       NV_API_CALL  rm_is_legacy_device     (U016, BOOL);
#endif /* NVWATCH */

#endif /* __KERNEL__ */

#endif
