/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 2001-2002 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */

#include "nv-misc.h"
#include "os-interface.h"
#include "nv.h"
#include "nv-freebsd.h"

static int nvidia_pci_probe  (device_t);
static int nvidia_pci_attach (device_t);
static int nvidia_pci_detach (device_t);

int nvidia_pci_probe(device_t dev)
{
    U016 device;
    char name[NV_DEVICE_NAME_LENGTH];

    if ((pci_get_class(dev) != PCIC_DISPLAY) ||
            (pci_get_subclass(dev) != PCIS_DISPLAY_VGA) ||
            (pci_get_vendor(dev) != 0x10de) ||
            ((device = pci_get_device(dev)) < 0x0020))
        return ENXIO;

    if (rm_is_legacy_device(device, TRUE)) {
        return ENXIO;
    }
    
    if (rm_get_device_name(NULL, device, NV_DEVICE_NAME_LENGTH, name)
            != RM_OK) {
        strcpy(name, "Unknown");
    }

    device_set_desc_copy(dev, name);
    return 0;
}

int nvidia_pci_setup_intr(device_t dev)
{
    int status, flags;
    struct nvidia_softc *sc;

    sc = device_get_softc(dev);

    /* XXX Revisit! (INTR_FAST, INTR_MPSAFE) */
    flags = INTR_TYPE_AV;

    status = bus_setup_intr(dev, sc->irq, flags, nvidia_intr, sc, &sc->irq_ih);
    if (status) {
        device_printf(dev, "NVRM: HW ISR setup failed.\n");
        goto fail;
    }

fail:
    return status;
}

int nvidia_pci_teardown_intr(device_t dev)
{
    int status;
    struct nvidia_softc *sc;

    sc = device_get_softc(dev);

    status = bus_teardown_intr(dev, sc->irq, sc->irq_ih);
    if (status) {
        device_printf(dev, "NVRM: HW ISR teardown failed.\n");
        goto fail;
    }

fail:
    return status;
}

void nvidia_pci_check_config_space(device_t dev)
{
    U032 BAR_low, BAR_high;
    U016 word, i;
    struct nvidia_softc *sc = device_get_softc(dev);
    nv_state_t *nv = sc->nv_state;

    word = os_pci_read_word(dev, PCIR_COMMAND);

    if ((word & PCIM_CMD_BUSMASTEREN) == 0)
        pci_enable_busmaster(dev);

    if ((word & PCIM_CMD_MEMEN) == 0)
        pci_enable_io(dev, SYS_RES_MEMORY);

    for (i = 0; i < NV_GPU_NUM_BARS; i++) {
        BAR_low = BAR_high = 0;
        nv_aperture_t *BAR = &nv->bars[i];
        if (BAR->offset == 0) continue;

        BAR_low = os_pci_read_dword(dev, BAR->offset);
        if ((BAR_low & NVRM_PCICFG_BAR_ADDR_MASK) != BAR->address)
            os_pci_write_dword(dev, BAR->offset, BAR->address);

        if ((BAR_low & NVRM_PCICFG_BAR_MEMTYPE_MASK)
                != NVRM_PCICFG_BAR_MEMTYPE_64BIT)
            continue;

        BAR_high = os_pci_read_dword(dev, BAR->offset + 4);
        if ((BAR_low & NVRM_PCICFG_BAR_ADDR_MASK) == BAR->address
                && BAR_high == 0)
            continue;

        os_pci_write_dword(dev, BAR->offset, BAR->address); /* 31:4 */
        os_pci_write_dword(dev, BAR->offset + 4, 0); /* 63:32 */
    }
}

int nvidia_pci_attach(device_t dev)
{
    int status;
    struct nvidia_softc *sc;
    U016 word, i, j;
    U032 BAR_low, req;

    sc = device_get_softc(dev); /* first reference */
    bzero(sc, sizeof(nvidia_softc_t));

    sc->nv_state = malloc(sizeof(nv_state_t), M_NVIDIA, M_WAITOK | M_ZERO);
    if (sc->nv_state == NULL)
        return ENOMEM;

    pci_enable_busmaster(dev);
    word = pci_read_config(dev, PCIR_COMMAND, 2);

    if ((word & PCIM_CMD_BUSMASTEREN) == 0) {
        device_printf(dev, "NVRM: PCI busmaster enable failed.\n");
        return ENXIO;
    }

    pci_enable_io(dev, SYS_RES_MEMORY);
    word = pci_read_config(dev, PCIR_COMMAND, 2);

    if ((word & PCIM_CMD_MEMEN) == 0) {
        device_printf(dev, "NVRM: PCI memory enable failed.\n");
        return ENXIO;
    }

    for (i = 0, j = 0; i < NVRM_PCICFG_NUM_BARS && j < NV_GPU_NUM_BARS; i++) {
        U008 offset = NVRM_PCICFG_BAR_OFFSET(i);
        BAR_low = os_pci_read_dword(dev, offset);
        os_pci_write_dword(dev, offset, 0xffffffff);
        req = os_pci_read_dword(dev, offset);
        if ((req != 0) /* implemented */ && (req & NVRM_PCICFG_BAR_REQTYPE_MASK)
                == NVRM_PCICFG_BAR_REQTYPE_MEMORY) {
            sc->nv_state->bars[j].offset = offset;
            sc->BAR_rids[j] = offset; j++;
            if ((req & NVRM_PCICFG_BAR_MEMTYPE_MASK) == NVRM_PCICFG_BAR_MEMTYPE_64BIT)
                i++;
        }
        os_pci_write_dword(dev, offset, BAR_low);
    }

    sc->irq_rid = 0;
    sc->iop_rid = 0;

    status = nvidia_alloc(dev);
    if (status) {
        device_printf(dev, "NVRM: NVIDIA driver alloc failed.\n");
        goto fail;
    }

    if (!rm_init_private_state(sc->nv_state)) {
        device_printf(dev, "NVRM: rm_init_private_state() failed.\n");
        goto fail;
    }

    status = nvidia_attach(dev);
    if (status) {
        rm_free_private_state(sc->nv_state);
        device_printf(dev, "NVRM: NVIDIA driver attach failed.\n");
        goto fail;
    }

    status = nvidia_pci_setup_intr(dev);
    if (status) {
        nvidia_detach(dev);
        rm_free_private_state(sc->nv_state);
        goto fail;
    }

#if __FreeBSD_version >= 500000
    mtx_init(&sc->mtx_rm, "dev.mtx_rm", NULL, MTX_SPIN | MTX_RECURSE);
    sx_init(&sc->sx_api, "dev.sx_api");
#else
    /*
     * Assume that none of the API paths are reentrant and
     * that any could could conceivably sleep.
     */
    lockinit(&sc->api_lock, PZERO | PCATCH, "dev.api_lock", 0, 0); 
#endif

    return 0;

fail:
    nvidia_free(dev);
    free(sc->nv_state, M_NVIDIA);
    return status;
}

int nvidia_pci_detach(device_t dev)
{
    int status;
    struct nvidia_softc *sc;

    /*
     * Check if the device is still in use before accepting the
     * detach request; this event can happen even when the module
     * usage count is non-zero!
     */
    sc = device_get_softc(dev);
    if (sc->refcnt != 0) /* XXX Fix me? (refcnt) */
        return EBUSY;

#if __FreeBSD_version >= 500000
    mtx_destroy(&sc->mtx_rm);
    sx_destroy(&sc->sx_api);
#else
    lockmgr(&sc->api_lock, LK_DRAIN, 0, CURTHREAD);
#endif

    status = nvidia_pci_teardown_intr(dev);
    if (status)
        goto fail;

    status = nvidia_detach(dev);
    if (status) {
        device_printf(dev, "NVRM: NVIDIA driver detach failed.\n");
        goto fail;
    }

    rm_free_private_state(sc->nv_state);
    nvidia_free(dev);
    free(sc->nv_state, M_NVIDIA);

fail:
    /* XXX Fix me? (state) */
    return status;
}

static device_method_t nvidia_pci_methods[] = {
    DEVMETHOD( device_probe,   nvidia_pci_probe  ),
    DEVMETHOD( device_attach,  nvidia_pci_attach ),
    DEVMETHOD( device_detach,  nvidia_pci_detach ),
#ifdef NV_SUPPORT_ACPI_PM
    DEVMETHOD( device_suspend, nvidia_suspend    ),
    DEVMETHOD( device_resume,  nvidia_resume     ),
#endif
    { 0, 0 }
};

static driver_t nvidia_pci_driver = {
    "nvidia",
    nvidia_pci_methods,
    sizeof(struct nvidia_softc)
};

DRIVER_MODULE(nvidia, pci, nvidia_pci_driver, nvidia_devclass, nvidia_modevent, 0);

#if __FreeBSD_version >= 502126
MODULE_DEPEND(nvidia, mem, 1, 1, 1);
MODULE_DEPEND(nvidia, io, 1, 1, 1);
#endif

#ifdef NV_SUPPORT_OS_AGP
MODULE_DEPEND(nvidia, agp, 1, 1, 1);
#endif

#ifdef NV_SUPPORT_LINUX_COMPAT /* (COMPAT_LINUX || COMPAT_LINUX32) */
MODULE_DEPEND(nvidia, linux, 1, 1, 1);
#endif

