/* _NVRM_COPYRIGHT_BEGIN_
 *
 * Copyright 2001-2002 by NVIDIA Corporation.  All rights reserved.  All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 *
 * _NVRM_COPYRIGHT_END_
 */

#include "nv-misc.h"
#include "os-interface.h"
#include "nv.h"
#include "nv-freebsd.h"

#ifdef NV_USE_OS_VM86_INT10CALL
#include <machine/vm86.h>
#endif

devclass_t nvidia_devclass;
nv_state_t nvidia_ctl_state;

int nvidia_attach(device_t dev)
{
    int status;
    U032 i;
    struct nvidia_softc *sc;
    nv_state_t *nv;

    sc = device_get_softc(dev);
    nv = sc->nv_state;

    nv->os_state         = sc;
    nv->flags            = 0;
    nv->bus              = pci_get_bus(dev);
    nv->slot             = pci_get_slot(dev);
    nv->vendor_id        = pci_get_vendor(dev);
    nv->device_id        = pci_get_device(dev);
    nv->interrupt_line   = pci_get_irq(dev);

    for (i = 0; i < NV_GPU_NUM_BARS; i++) {
        if (sc->BAR_recs[i] != NULL) {
            nv->bars[i].address = rman_get_start(sc->BAR_recs[i]);
            nv->bars[i].size = rman_get_size(sc->BAR_recs[i]);
        }
    }

    nv->fb   = &nv->bars[NV_GPU_BAR_INDEX_FB];
    nv->regs = &nv->bars[NV_GPU_BAR_INDEX_REGS];

    if ((status = nvidia_dev_attach(sc)) != 0)
        return status;

    if ((status = nvidia_ctl_attach()) != 0)
        return status;

    nv_sysctl_init(nv);
    return 0;
}

int nvidia_detach(device_t dev)
{
    int status;
    struct nvidia_softc *sc;

    sc = device_get_softc(dev);
    nv_sysctl_exit(sc->nv_state);

    status = nvidia_dev_detach(sc);
    if (status) {
        device_printf(dev, "NVRM: NVIDIA driver DEV detach failed.\n");
        goto fail;
    }

    status = nvidia_ctl_detach();
    if (status) {
        device_printf(dev, "NVRM: NVIDIA driver CTL detach failed.\n");
        goto fail;
    }

fail:
    /* XXX Fix me? (state) */
    return status;
}


#ifdef NV_SUPPORT_ACPI_PM
int nvidia_suspend(device_t dev)
{
    struct nvidia_softc *sc;
    nv_state_t *nv;
    int status = RM_ERROR;

    /* Only if ACPI is running */
    if (devclass_get_softc(devclass_find("ACPI"), 0) == NULL)
        return ENODEV;

    sc = device_get_softc(dev);
    nv = sc->nv_state;

    NV_PCI_CHECK_CONFIG_SPACE(nv);
    status = rm_power_management(nv, 0, NV_PM_ACPI_STANDBY);

    return status;
}

int nvidia_resume(device_t dev)
{
    struct nvidia_softc *sc;
    nv_state_t *nv;
    int status = RM_ERROR;

    sc = device_get_softc(dev);
    nv = sc->nv_state;

    NV_PCI_CHECK_CONFIG_SPACE(nv);
    status = rm_power_management(nv, 0, NV_PM_ACPI_RESUME);

    return status;
}
#endif /* NV_SUPPORT_ACPI_PM */


int nvidia_alloc(device_t dev)
{
    int error = 0;
    struct nvidia_softc *sc;
    U032 flags, i;

    sc = device_get_softc(dev);
    sc->dev = dev;

    flags = 0; /* not RF_ACTIVE */
    for (i = 0; i < NV_GPU_NUM_BARS && sc->BAR_rids[i] != 0; i++) {
        struct resource *res;
        res = bus_alloc_resource_any(dev, SYS_RES_MEMORY, &sc->BAR_rids[i], flags);
        if (res == NULL) {
            /*
             * The most likely reason for this failure is that the SBIOS failed
             * to assign a valid address range to this BAR; FreeBSD is unable to
             * correct the problem and fails this BUS resource allocation. We
             * trust the kernel with BAR validation at this point, but later try
             * to catch cases where the X server "corrects" "invalid" BAR's.
             *
             * Please see to nvidia_pci_check_config_space() in nvidia_pci.c for
             * additional information.
             */
            device_printf(dev,
                "NVRM: NVIDIA MEM resource alloc failed, BAR%d @ 0x%02x.\n",
                i, sc->nv_state->bars[i].offset);
            error = ENXIO;
            goto fail;
        }
        sc->BAR_recs[i] = res;
    }

    flags = RF_SHAREABLE | RF_ACTIVE;
    sc->irq = bus_alloc_resource_any(dev, SYS_RES_IRQ, &sc->irq_rid, flags);
    if (sc->irq == NULL) {
        device_printf(dev, "NVRM: NVIDIA IRQ resource alloc failed.\n");
        error = ENXIO;
        goto fail;
    }

fail:
    return (error);
}

void nvidia_free(device_t dev)
{
    struct nvidia_softc *sc;
    U032 i;

    sc = device_get_softc(dev);

    for (i = 0; i < NV_GPU_NUM_BARS && sc->BAR_recs[i] != NULL; i++)
        bus_release_resource(dev, SYS_RES_MEMORY, sc->BAR_rids[i], sc->BAR_recs[i]);
    if (sc->irq != NULL)
        bus_release_resource(dev, SYS_RES_IRQ, sc->irq_rid, sc->irq);
    if (sc->iop != NULL)
        bus_release_resource(dev, SYS_RES_IOPORT, sc->iop_rid, sc->iop);
}

void nvidia_intr(void *xsc)
{
    struct nvidia_softc *sc;
    nv_state_t *nv;
    U032 run_bottom_half = 0;

    sc = (struct nvidia_softc *) xsc;
    nv = sc->nv_state;

    NV_PCI_CHECK_CONFIG_SPACE(nv);
    rm_isr(nv, &run_bottom_half);

    if (run_bottom_half) {
        /* We're not executing in an HW ISR context */
        rm_isr_bh(nv);
    }
}

int nvidia_post_vbios(struct nv_ioctl_post_vbios *pv)
{
    unsigned int i;
    struct nvidia_softc *sc;
    nv_state_t *nv;

    for (i = 0; i < NV_MAX_DEVICES; i++) {
        sc = devclass_get_softc(nvidia_devclass, i);
        if (!sc)
            continue;
        nv = sc->nv_state;

        if ((nv->bus == pv->bus) && (nv->slot == pv->slot) &&
                (nv->flags & NV_FLAG_WAS_POSTED) == 0)
            nv->flags |= NV_FLAG_NEEDS_POSTING;
    }

    return 0;
}

int nvidia_get_api_version(struct nv_ioctl_rm_api_version *av)
{
    av->major   = NV_MAJOR_VERSION;
    av->minor   = NV_MINOR_VERSION;
    av->patch   = NV_PATCHLEVEL;

    return 0;
}

int nvidia_get_card_info(struct nv_ioctl_card_info *ci)
{
    unsigned int i;
    struct nvidia_softc *sc;
    nv_state_t *nv;

    /*
     * Clients supporting versioning will pass version magic in the first
     * card information field.
     */
    struct nv_ioctl_rm_api_version *av = (void *) ci;
    int status = 0;

    switch (av->magic) {
        case NV_RM_API_VERSION_MAGIC_OVERRIDE_REQ:
            break;
        case NV_RM_API_VERSION_MAGIC_LAX_REQ:
        case NV_RM_API_VERSION_MAGIC_REQ:
            if ((av->major != NV_MAJOR_VERSION) ||
                (av->minor != NV_MINOR_VERSION) ||
                (av->magic == NV_RM_API_VERSION_MAGIC_REQ ?
                    (av->patch != NV_PATCHLEVEL) :
                    (av->patch / 500 != NV_PATCHLEVEL / 500))) {
                printf("NVRM: version mismatch: "
                        "RM %d.%d-%d, client %d.%d-%d\n",
                        NV_MAJOR_VERSION, NV_MINOR_VERSION,
                        NV_PATCHLEVEL,
                        av->major, av->minor, av->patch);
                status = -EINVAL;
            }
            break;
        default:
            status = -EINVAL;
            break;
    }

    if (status != 0) {
        /*
         * This client either has no versioning support its version isn't
         * identical to ours. Either scenario is dangerous, as we cannot
         * guarantee that data structures passed between client and RM are
         * identical, or that the RM performs the expected operation(s).
         */
        av->magic   = NV_RM_API_VERSION_MAGIC_REP;
        av->major   = NV_MAJOR_VERSION;
        av->minor   = NV_MINOR_VERSION;
        av->patch   = NV_PATCHLEVEL;

        return status;
    }

    /* clear card information structure */
    memset(ci, 0, sizeof(ci));

    for (i = 0; i < NV_MAX_DEVICES; i++) {
        sc = devclass_get_softc(nvidia_devclass, i);
        if (!sc)
            continue;
        nv = sc->nv_state;

        ci[i].flags          = (NV_IOCTL_CARD_INFO_FLAG_PRESENT |
                                NV_IOCTL_CARD_INFO_FLAG_NEED_MSYNC);
        ci[i].bus            = nv->bus;
        ci[i].slot           = nv->slot;
        ci[i].vendor_id      = nv->vendor_id;
        ci[i].device_id      = nv->device_id;
        ci[i].interrupt_line = nv->interrupt_line;
        ci[i].fb_address     = nv->fb->address;
        ci[i].fb_size        = nv->fb->size;
        ci[i].reg_address    = nv->regs->address;
        ci[i].reg_size       = nv->regs->size;
    }

    return status;
}


int nvidia_handle_ioctl(
    dev_t dev,
    u_long cmd,
    caddr_t data,
    int fflag,
    d_thread_t *td
)
{
    struct nvidia_softc *sc;
    nv_state_t *nv;
    int unit = minor(dev);

    if (unit == CDEV_CTL_MINOR) {
        /* the control device is "special" */
        nv = &nvidia_ctl_state;
    } else {
        sc = devclass_get_softc(nvidia_devclass, unit);
        if (!sc)
            return ENXIO;
        nv = sc->nv_state;
    }

    NV_PCI_CHECK_CONFIG_SPACE(nv);

    if (rm_ioctl(nv, __TD_FDT(td), __NV_IOC_NR(cmd), data))
        return 0;

    return EINVAL;
}


int nvidia_open_ctl(void)
{
    struct nvidia_softc *sc;
    nv_state_t *nv = &nvidia_ctl_state;

    sc = nv->os_state;
    sc->refcnt++;

    nv->flags |= NV_FLAG_OPEN;
    nv->flags |= NV_FLAG_CONTROL;

    nv->flags &= ~NV_FLAG_HOTKEY_OCCURRED;

    return 0;
}

int nvidia_close_ctl(
    dev_t dev,
    d_thread_t *td
)
{
    struct nvidia_softc *sc;
    nv_state_t *nv = &nvidia_ctl_state;

    if (__TD_FDT_CNT(td) == 0)
        rm_free_unused_clients(nv, 0, __TD_FDT(td));

    sc = nv->os_state;
    sc->refcnt--;

    if (sc->refcnt == 0) {
        /*
         * The control device has been released; without physical devices
         * backing it, we only need to reset the flags.
         */
        nv->flags = 0;
    }

    return 0;
}

int nvidia_open_dev(struct nvidia_softc *sc)
{
    nv_state_t *nv = sc->nv_state;

    NV_PCI_CHECK_CONFIG_SPACE(nv);

    if ((nv->flags & NV_FLAG_OPEN) == 0) {
        /*
         * The device flags indicate that this device has not been opened
         * or else shut down previously. There isn't much we can do here,
         * the core resource manager does the actual work. We update both
         * the flags and the usage count.
         */
        STAILQ_INIT(&sc->event_queue);

        if (!rm_init_adapter(nv)) {
            device_printf(sc->dev, "NVRM: rm_init_adapter() failed!\n");
            return EIO;
        }

        nv->flags |= NV_FLAG_OPEN;
    }

    sc->refcnt++;

    return 0;
}

int nvidia_close_dev(
    struct nvidia_softc *sc,
    dev_t dev,
    d_thread_t *td
)
{
    nv_state_t *nv = sc->nv_state;
    nv_os_event_t *et;

    NV_PCI_CHECK_CONFIG_SPACE(nv);

    if (__TD_FDT_CNT(td) == 0)
        rm_free_unused_clients(nv, 0, __TD_FDT(td));

    sc->refcnt--;

    if (sc->refcnt == 0) {
        /*
         * The usage count for this device has dropped to zero, it can be
         * safely shut down. We don't need to wait for bottom-halfes like
         * we do on Linux, they are not run asynchronously on FreeBSD. We
         * do need to reset certain flags, though.
         */
        rm_disable_adapter(nv);
        rm_shutdown_adapter(nv);

        while ((et = STAILQ_FIRST(&sc->event_queue))) {
            STAILQ_REMOVE(&sc->event_queue, et, nv_os_event, queue);
            free(et, M_NVIDIA);
        }

        nv->flags &= ~NV_FLAG_OPEN;
    }

    return 0;
}


int nvidia_modevent(
    module_t mod,
    int what,
    void *arg
)
{
    nv_state_t *nv;
    struct nvidia_softc *sc;

    switch (what) {
        case MOD_LOAD:
            /*
             * The module load event. Our KLD has just been loaded and is
             * ready to initialize. We setup the core resource manager in
             * this routine, further initialization takes place at attach
             * time.
             */
            if (!rm_init_rm()) {
                printf("NVRM: rm_init_rm() failed!\n");
                return EIO;
            }

            nvidia_sysctl_init();
            nvidia_linux_init();

            break;

        case MOD_UNLOAD:
            /*
             * Check if the control device is still open and reject the
             * unload request if it is. This event can occur even when the
             * module usage count is non-zero!
             */
            nv = &nvidia_ctl_state;
            sc = nv->os_state;

            if (sc->refcnt != 0) /* XXX Fix me? (refcnt) */
                return EBUSY;

            rm_shutdown_rm();

            nvidia_sysctl_exit();
            nvidia_linux_exit();

            break;

        default:
            break;
    }

    return 0;
}


#ifdef NV_SUPPORT_OS_AGP
S032 nv_os_agp_init(
    nv_state_t *nv,
    void **base,
    void **linear,
    U032 *limit
)
{
    void *bitmap;
    struct nvidia_softc *sc = nv->os_state;
    struct agp_info ai;

    U032 mode = 0;
    U032 fw   = 0;
    U032 sba  = 0;
    U032 rate = AGP_MODE_RATE_1x | AGP_MODE_RATE_2x | AGP_MODE_RATE_4x;
    U032 size = 0;

    sc->agp_dev = agp_find_device();
    if (!sc->agp_dev) {
        printf("NVRM: agp_find_device failed, chipset unsupported?\n");
        return -ENODEV;
    }

    if (agp_acquire(sc->agp_dev) != 0)
        return -EBUSY;

    agp_get_info(sc->agp_dev, &ai);
    mode = ai.ai_mode;

    if (os_set_mem_range(ai.ai_aperture_base, ai.ai_aperture_size,
                NV_MEMORY_WRITECOMBINED) != RM_OK) {
        /*
         * Failure to set a write-combining range for the AGP aperture is
         * not necessarily a fatal error condition; we don't know at this
         * point, however, and abort to prevent performance and stability
         * problems that may be hard to track down otherwise.
         */
        agp_release(sc->agp_dev);
        return -EIO;
    }

    rm_read_registry_dword(NULL, "NVreg", "ReqAGPRate", &rate);
    rm_read_registry_dword(NULL, "NVreg", "EnableAGPFW", &fw);
    rm_read_registry_dword(NULL, "NVreg", "EnableAGPSBA", &sba);

    mode = AGP_MODE_SET_RATE(mode, AGP_MODE_GET_RATE(mode) & rate);
    mode = AGP_MODE_SET_FW(mode, fw);
    mode = AGP_MODE_SET_SBA(mode, sba);

    if (agp_enable(sc->agp_dev, mode) != 0) {
        agp_release(sc->agp_dev);
        os_unset_mem_range(ai.ai_aperture_base, ai.ai_aperture_size);
        return -EIO;
    }

    size = ai.ai_aperture_size / RM_PAGE_SIZE / 8;
    
    if (os_alloc_mem(&bitmap, size) != RM_OK) {
        agp_release(sc->agp_dev);
        os_unset_mem_range(ai.ai_aperture_base, ai.ai_aperture_size);
        return -EIO;
    }

    os_mem_set(bitmap, 0xff, size);

    if (rm_set_agp_bitmap(nv, bitmap) != RM_OK) {
        agp_release(sc->agp_dev);
        os_free_mem(bitmap);
        os_unset_mem_range(ai.ai_aperture_base, ai.ai_aperture_size);
        return -EIO;
    }

    *base   = (void *) ai.ai_aperture_base;
    *linear = (void *) ai.ai_aperture_va;
    *limit  = (U032)   ai.ai_aperture_size - 1;

    return 0;
}

S032 nv_os_agp_teardown(nv_state_t *nv)
{
    struct nvidia_softc *sc = nv->os_state;
    void *bitmap;

    if (agp_release(sc->agp_dev) != 0)
        return -EBUSY;

    rm_clear_agp_bitmap(nv, &bitmap);
    os_free_mem(bitmap);

    os_unset_mem_range(nv->agp.address, nv->agp.size);

    return 0;
}
#endif /* NV_SUPPORT_OS_AGP */

S032 nv_agp_init(
    nv_state_t *nv,
    void **base,
    void **linear,
    void *limit,
    U032 config
)
{
    if (NV_AGP_ENABLED(nv))
        return 1;

    if (config == NVOS_AGP_CONFIG_DISABLE_AGP) {
        /*
         * Match the behavior on Linux, don't consider the attempt
         * to initialize AGP as 'disabled' an error.
         */
        nv->agp_config = NVOS_AGP_CONFIG_DISABLE_AGP;
        nv->agp_status = NV_AGP_STATUS_DISABLED;
        return 0;
    }

#ifdef NV_SUPPORT_OS_AGP
    if ((config & NVOS_AGP_CONFIG_OSAGP) != 0) {
        if (nv_os_agp_init(nv, base, linear, limit) == 0) {
            /*
             * If the operating system AGP GART driver successfully
             * configured its backend, apply chipset overrides.
             */
            rm_update_agp_config(nv);
            nv->agp_config = NVOS_AGP_CONFIG_OSAGP;
            nv->agp_status = NV_AGP_STATUS_ENABLED;
            return 0;
        }
    }
#endif /* NV_SUPPORT_OS_AGP */

    if ((config & NVOS_AGP_CONFIG_NVAGP) == 0)
        goto failed;

    if (devclass_get_softc(devclass_find("agp"), 0) != NULL) {
        /*
         * Make sure we don't try to use the internal GART driver when
         * the OS AGPGART driver (agp.ko) is attached. While that may
         * be perfectly fine on most systems, but is known to break on
         * some.
         * -------------------------------------------------------------
         * DON'T REDISTRIBUTE THE DRIVER WITH THIS SANITY CHECK REMOVED!
         * -------------------------------------------------------------
         */
        printf("NVRM: detected agp.ko, aborting NVIDIA AGP setup!\n");
        goto failed;
    }

    if (rm_init_agp(nv) == RM_OK) {
        nv->agp_config = NVOS_AGP_CONFIG_NVAGP;
        nv->agp_status = NV_AGP_STATUS_ENABLED;
        return 0;
    }

failed:
    nv->agp_config = NVOS_AGP_CONFIG_DISABLE_AGP;
    nv->agp_status = NV_AGP_STATUS_FAILED;
    return -EIO;
}

S032 nv_agp_teardown(nv_state_t *nv)
{
    S032 status = -EINVAL;

    if (!NV_AGP_ENABLED(nv))
        return status;

#ifdef NV_SUPPORT_OS_AGP
    if (NV_OSAGP_ENABLED(nv))
        status = nv_os_agp_teardown(nv);
#endif
    if (NV_NVAGP_ENABLED(nv))
        status = rm_teardown_agp(nv);

    nv->agp_config = NVOS_AGP_CONFIG_DISABLE_AGP;
    nv->agp_status = NV_AGP_STATUS_DISABLED;

    return status;
}

S032 nv_no_incoherent_mappings(void)
{
    /* XXX Fix me! (no_incoherent_mappings) */
    return 0;
}

void* nv_find_agp_mapping(
    nv_state_t *nv,
    ULONG address
)
{
    /* XXX Implement me?  (find_agp_mapping) */
    return NULL;
}

void* nv_find_agp_kernel_mapping(
    nv_state_t *nv,
    ULONG address
)
{
    /* XXX Implement me? (find_agp_kernel_mapping) */
    return NULL;
}


void nv_lock_rm(nv_state_t *nv)
{
#if __FreeBSD_version >= 500000
    /*
     * With SMPng, the "giant" kernel lock is gone. That means that we're
     * in a more complex enviroment locking-wise, but since the necessary
     * locking primitives are available to us, we can handle it.
     *
     * With mtx_lock_spin we acquire a spin mutex and locally disable all
     * interrupts on the current processor.
     */
    struct nvidia_softc *sc = nv->os_state;
    mtx_lock_spin(&sc->mtx_rm);
#else
    /* 
     * While SMP configurations are handled with a global kernel lock and
     * in such a way that only one of the CPUs is executing a process and
     * servicing interrupts at any point in time, it is possible that the
     * interrupt handlers (th/bh) may interfere with critical code.
     *
     * This "lock" thus disables interrupts for the duration of this code
     * segment's execution.
     */
    struct nvidia_softc *sc = nv->os_state;
    sc->spl = splhigh();
#endif
}

void nv_unlock_rm(nv_state_t *nv)
{
#if __FreeBSD_version >= 500000
    struct nvidia_softc *sc = nv->os_state;
    mtx_unlock_spin(&sc->mtx_rm);
#else
    struct nvidia_softc *sc = nv->os_state;
    splx(sc->spl);
#endif
}

void nv_lock_api(nv_state_t *nv)
{
#if __FreeBSD_version >= 500000
    struct nvidia_softc *sc = nv->os_state;
    sx_xlock(&sc->sx_api);
#else
    struct nvidia_softc *sc = nv->os_state;
    lockmgr(&sc->api_lock, LK_EXCLUSIVE, 0, CURTHREAD);
#endif
}

void nv_unlock_api(nv_state_t *nv)
{
#if __FreeBSD_version >= 500000
    struct nvidia_softc *sc = nv->os_state;
    sx_xunlock(&sc->sx_api);
#else
    struct nvidia_softc *sc = nv->os_state;
    lockmgr(&sc->api_lock, LK_RELEASE, 0, CURTHREAD);
#endif
}


void nv_post_event(
    nv_state_t *nv,
    nv_event_t *event,
    U032 hObject,
    U032 index
)
{
    struct nvidia_softc *sc;
    nv_os_event_t *et; 

    if (!(et = malloc(sizeof(nv_os_event_t), M_NVIDIA, M_NOWAIT | M_ZERO)))
        return;

    et->event = *event;
    et->event.hObject = hObject;
    et->event.index = index;

    nv_lock_rm(nv);

    sc = nv->os_state;
    STAILQ_INSERT_TAIL(&sc->event_queue, et, queue);

    nv_unlock_rm(nv);

/*  printf("NVRM: nv_post_event()\n");
    printf("NVRM:     hParent: 0x%08x\n", et->event.hParent);
    printf("NVRM:     hObject: 0x%08x\n", et->event.hObject); */

    /* XXX Fix me? (os events) */
    selwakeup(&sc->rsel);
}

S032 nv_get_event(
    nv_state_t *nv,
    void *file,
    nv_event_t *event,
    U032 *pending
)
{
    struct nvidia_softc *sc = nv->os_state;
    nv_os_event_t *et;

    nv_lock_rm(nv);

    if (!(et = STAILQ_FIRST(&sc->event_queue))) {
        nv_unlock_rm(nv);
        return RM_ERROR; /* RM polling? */
    }

    *event = et->event;

    STAILQ_REMOVE_HEAD(&sc->event_queue, queue);

/*  printf("NVRM: nv_get_event(), %d\n", *pending);
    printf("NVRM:     hParent: 0x%08x\n", event->hParent);
    printf("NVRM:     hObject: 0x%08x\n", event->hObject); */
    
    /* XXX: Fix me? (os events) */
    *pending = !(STAILQ_EMPTY(&sc->event_queue));

    nv_unlock_rm(nv);

    /* will attempt to acquire a blockable sleep lock */
    free(et, M_NVIDIA);

    return RM_OK;
}

void* nv_find_alloc(
    nv_state_t *nv,
    vm_offset_t address
)
{
    nv_alloc_t *at;
    struct nvidia_softc *sc = nv->os_state;

    SLIST_FOREACH(at, &sc->alloc_list, list) {
        /*
         * XXX Match an incoming address to a known allocation. We
         * should use type flags here.
         */
        if (at->address == address)
            return (void *) at;
    }

    return NULL;
}

void* nv_find_alloc_obj(
    nv_state_t *nv,
    vm_offset_t object
)
{
    nv_alloc_t *at;
    struct nvidia_softc *sc = nv->os_state;

    SLIST_FOREACH(at, &sc->alloc_list, list) {
        /*
         * XXX Match the incoming object to a known allocation. We
         * should use type flags here.
         */
        if (at->object == (vm_object_t) object)
            return (void *) at;
    }

    return NULL;
}

void* nv_find_nv_mapping(
    nv_state_t *nv,
    ULONG address
)
{
    nv_alloc_t *at;
    struct nvidia_softc *sc = nv->os_state;
    vm_offset_t offset, linear;

    offset = (vm_offset_t) address & PAGE_MASK;
    address &= ~PAGE_MASK;

    SLIST_FOREACH(at, &sc->alloc_list, list) {
        /*
         * XXX Match an incoming address to a known allocation. We
         * should use type flags here.
         */
        if (at->address == 0)
            continue;
        linear = at->address;
        do {
            if (vtophys(linear) == (vm_offset_t) address)
                return (void *)(linear + offset);
            linear += PAGE_SIZE;
        } while (linear < (at->address + at->size));
    }

    return NULL;
}

ULONG nv_find_dma_mapping(
    nv_state_t *nv,
    ULONG address,
    void **track,
    U032 rm_page_hint
)
{
    /* XXX Fix me? (x86 only, amd64 dma) */
    return address;
}

/* Lookup the physiscal address of the page that backs a dma address */
ULONG  nv_dma_to_phys_address(
    nv_state_t *nv,
    ULONG dma_address
)
{
    /* 
     * Our FreeBSD driver currently makes no distinction between
     * physical and dma addresses.
     */
    return dma_address; 
}

S032 nv_translate_address(
    nv_state_t *nv,
    unsigned long base,
    U032 index,
    unsigned long *address
)
{
    nv_alloc_t *at;
    struct nvidia_softc *sc = nv->os_state;

    u_int32_t count;
    vm_page_t m;
    vm_offset_t vm = NV_MMAP_TO_VM_OFFSET(base);

    SLIST_FOREACH(at, &sc->alloc_list, list) {
        count = (at->size / PAGE_SIZE);
        /* AGP memory */
        if (at->object == (vm_object_t) base && index < count) {
            VM_OBJECT_LOCK(at->object);
            m = vm_page_lookup(at->object, index);
            *address = VM_PAGE_TO_PHYS(m);
            VM_OBJECT_UNLOCK(at->object);
            return 0;
        }
        /* PCI memory (user) */
        if (at->address == (vm_offset_t) vm && index < count) {
            vm = at->address + (index * PAGE_SIZE);
            *address = vtophys(vm);
            return 0;
        }
        /* PCI memory (NVRM) */
        if (at->address == (vm_offset_t) base && index < count) {
            vm = at->address + (index * PAGE_SIZE);
            *address = vtophys(vm);
            return 0;
        }
    }

    return -1;
}

S032 nv_alloc_contig_pages(
    nv_state_t *nv,
    void **address,
    U032 count
)
{
    nv_alloc_t *at;
    struct nvidia_softc *sc = nv->os_state;
    U032 size = count * PAGE_SIZE;

    if (os_alloc_contig_pages(address, size) != RM_OK)
        return -ENOMEM;

    at = malloc(sizeof(nv_alloc_t), M_NVIDIA, M_WAITOK | M_ZERO);
    if (!at) {
        os_free_contig_pages(*address, size);
        return -ENOMEM;
    }

    at->object = 0;
    at->size = size;
    at->address = (vm_offset_t) *address;
    at->offset = ~0;

    SLIST_INSERT_HEAD(&sc->alloc_list, at, list);

    return 0;
}

S032 nv_free_contig_pages(
    nv_state_t *nv,
    void *address
)
{
    nv_alloc_t *at;
    struct nvidia_softc *sc = nv->os_state;
    U032 size;

    at = nv_find_alloc(nv, (vm_offset_t) address);
    if (!at) {
        os_dbg_breakpoint();
        return -EINVAL;
    }

    size = at->size;
    SLIST_REMOVE(&sc->alloc_list, at, nv_alloc, list);

    free(at, M_NVIDIA);
    os_free_contig_pages(address, size);

    return 0;
}

S032 nv_alloc_system_pages(
    nv_state_t  *nv,
    void **address,
    U032 count
)
{
    nv_alloc_t *at;
    struct nvidia_softc *sc = nv->os_state;

    void *vm;
    u_int32_t i, size = count * PAGE_SIZE;

    at = malloc(sizeof(nv_alloc_t), M_NVIDIA, M_WAITOK | M_ZERO);
    if (!at) {
        return -ENOMEM;
    }

    vm = malloc(size, M_NVIDIA, M_WAITOK | M_ZERO);
    if (!vm) {
        free(at, M_NVIDIA);
        return -ENOMEM;
    }

    at->pte = malloc(sizeof(vm_offset_t) * count, M_NVIDIA, M_WAITOK | M_ZERO);
    if (!at->pte) {
        free(at, M_NVIDIA);
        free(vm, M_NVIDIA);
        return -ENOMEM;
    }

    for (i = 0; i < count; i++) {
        at->pte[i] = vtophys((vm_offset_t) vm + (i * PAGE_SIZE));
        vm_page_lock_queues();
        vm_page_wire(PHYS_TO_VM_PAGE(at->pte[i]));
        vm_page_unlock_queues();
    }

    at->object = 0;
    at->size = size;
    at->address = (vm_offset_t) vm;
    at->offset = ~0;

    SLIST_INSERT_HEAD(&sc->alloc_list, at, list);
    *address = (void *) at->address;

    return 0;
}

S032 nv_free_system_pages(
    nv_state_t *nv,
    void *address
)
{
    nv_alloc_t *at;
    struct nvidia_softc *sc = nv->os_state;

    vm_offset_t pa;
    u_int32_t i, count;

    at = nv_find_alloc(nv, (vm_offset_t) address);
    if (!at) {
        os_dbg_breakpoint();
        return -EINVAL;
    }

    count = at->size / PAGE_SIZE;
    SLIST_REMOVE(&sc->alloc_list, at, nv_alloc, list);

    for (i = 0; i < count; i++) {
        pa = vtophys(at->address + (i * PAGE_SIZE));
        vm_page_lock_queues();
        vm_page_unwire(PHYS_TO_VM_PAGE(pa), 0);
        vm_page_unlock_queues();
    }

    free(at->pte, M_NVIDIA);
    free(at, M_NVIDIA);
    free(address, M_NVIDIA);

    return 0;
}


S032 nv_alloc_vm_object(
    nv_state_t *nv,
    void **address,
    U032 count
)
{
    nv_alloc_t *at;
    struct nvidia_softc *sc = nv->os_state;

    u_int32_t i, j;
    vm_page_t m;
    vm_object_t object;

    at = malloc(sizeof(nv_alloc_t), M_NVIDIA, M_WAITOK | M_ZERO);
    if (!at) {
        return -ENOMEM;
    }

    object = vm_object_allocate(OBJT_DEFAULT, count);
    if (!object) {
        free(at, M_NVIDIA);
        return -ENOMEM;
    }

    VM_OBJECT_LOCK(object);
    for (i = 0; i < count; i++) {
        /*
         * Find the individual pages belonging to the allocation from the
         * VM object; since this is the first time they're accessed, they
         * will be allocated.
         */

#if __FreeBSD_version >= 500000
        /*
         * Request that this range of pages be wired down to prevent them
         * from being paged out. This is done below for FreeBSD 4.
         */
        m = vm_page_grab(object, i, VM_ALLOC_NORMAL | VM_ALLOC_WIRED | VM_ALLOC_RETRY);
#else
        m = vm_page_grab(object, i, VM_ALLOC_NORMAL);
#endif
        if (!m) {
            for (j = 0; j < i; j++) {
                m = vm_page_lookup(object, j);
                vm_page_lock_queues();
                vm_page_unwire(m, 0);
                vm_page_unlock_queues();
            }
            VM_OBJECT_UNLOCK(object);
            free(at, M_NVIDIA); /* release pre-allocated resource(s) */
            return -ENOMEM;
        }

#if __FreeBSD_version < 500000
        vm_page_wire(m);
#endif
        vm_page_lock_queues();
        vm_page_wakeup(m);
        vm_page_unlock_queues();
    }
    VM_OBJECT_UNLOCK(object);

    at->object = object;
    at->offset = ~0;
    at->size = ptoa(count);
    at->address = 0;

    SLIST_INSERT_HEAD(&sc->alloc_list, at, list);
    *address = (void *) at->object;

    return 0;
}

S032 nv_free_vm_object(
    nv_state_t *nv,
    void *address
)
{
    nv_alloc_t *at;
    struct nvidia_softc *sc = nv->os_state;

    vm_page_t m;
    u_int32_t i, count;

    at = nv_find_alloc_obj(nv, (vm_offset_t) address);
    if (!at) {
        os_dbg_breakpoint();
        return -EINVAL;
    }

    count = at->size / PAGE_SIZE;
    SLIST_REMOVE(&sc->alloc_list, at, nv_alloc, list);

    VM_OBJECT_LOCK(at->object);
    for (i = 0; i < count; i++) {
        /*
         * Release "one wiring" of this page; this may reenable paging as
         * a result, depending on the total number of "wirings".
         */
        m = vm_page_lookup(at->object, i);
        vm_page_lock_queues();
        vm_page_unwire(m, 0);
        vm_page_unlock_queues();
    }
    VM_OBJECT_UNLOCK(at->object);

    vm_object_deallocate(at->object);
    free(at, M_NVIDIA);

    return 0;
}


#ifdef NV_SUPPORT_OS_AGP
S032 nv_alloc_agp_pages(
    nv_state_t *nv,
    void **address,
    U032 count,
    U032 offset,
    void **private
)
{
    void *handle;
    struct nvidia_softc *sc = nv->os_state;

    handle = agp_alloc_memory(sc->agp_dev, 0, count << PAGE_SHIFT);
    if (!handle) {
        /*
         * This is very unlikely to happen, the system's memory resources
         * would have to be nearly exhausted.
         */
        return -ENOMEM;
    }

    if (agp_bind_memory(sc->agp_dev, handle, offset) != 0) {
        /*
         * This shouldn't happen, we claimed the AGP backend and are thus
         * using it exclusively; the resource manager manages AGP offsets
         * internally, we wouldn't have been called had we run out of AGP
         * aperture space.
         */
        os_dbg_breakpoint();

        agp_free_memory(sc->agp_dev, handle);
        return -ENOMEM;
    }

    *private = handle;
    return 0;
}

S032 nv_free_agp_pages(
    nv_state_t *nv,
    void **address,
    U032 count,
    void *handle
)
{
    struct nvidia_softc *sc = nv->os_state;

    if (agp_unbind_memory(sc->agp_dev, handle) != 0) {
        /*
         * This is the only place where previously bound AGP memory would
         * be freed. If we fail to unbind this memory now, something very
         * wrong must have happened.
         */
        os_dbg_breakpoint();
    }

    agp_free_memory(sc->agp_dev, handle);
    return 0;
}
#endif /* NV_SUPPORT_OS_AGP */


S032 nv_alloc_pages(
    nv_state_t *nv,
    void **address,
    U032 count,
    U032 alloc_type_agp,
    U032 alloc_type_contiguous,
    U032 cache,
    U032 alloc_type_kernel,
    U032 class,
    void **private
)
{
    /* XXX Fix me! (cache types) */
    U032 alloc_type_cached = (cache != NV_MEMORY_UNCACHED);

    if (alloc_type_agp) {
        if (!NV_AGP_ENABLED(nv))
            return -EINVAL;

#ifdef NV_SUPPORT_OS_AGP
        if (NV_OSAGP_ENABLED(nv)) {
            U032 offset;

            if (rm_alloc_agp_bitmap(nv, count, &offset) != RM_OK) {
                /*
                 * We've run out of AGP aperture space, the requestor will be
                 * forced to fall back to general system memory.
                 */
                return -ENOMEM;
            }

            if (nv_alloc_agp_pages(nv, address, count, offset << PAGE_SHIFT,
                    private) != 0) {
                /*
                 * The AGP aperture is managed internally by core resman, the
                 * offset passed to us should work; thus, this shouldn't fail
                 * unless the system memory is exhausted.
                 */
                rm_free_agp_bitmap(nv, offset, count);
                return -ENOMEM;
            }

            *address = (void *)(NvUPtr)(nv->agp.address + (offset << PAGE_SHIFT));
            return 0;
        }
#endif /* NV_SUPPORT_OS_AGP */

        if (NV_NVAGP_ENABLED(nv)) {
            nv_alloc_t *at;

            if (nv_alloc_vm_object(nv, address, count) != 0) {
                /*
                 * Hopefully, this error condition will be much more unlikely
                 * to occur than failure to allocate the same number of pages
                 * with nv_alloc_system_pages.
                 */
                return -ENOMEM;
            }

            at = nv_find_alloc_obj(nv, (vm_offset_t) *address);
            if (!at) {
                os_dbg_breakpoint();
                return -ENOMEM;
            }

            if (rm_alloc_agp_pages(nv, address, count, class, private,
                    &at->offset) != RM_OK) {
                /*
                 * In theory, this should only fail if we run out of aperture
                 * space (which is not unlikely).
                 */
                nv_free_vm_object(nv, *address);
                return -ENOMEM;
            }

            *address = (void *)(NvUPtr)(nv->agp.address + (at->offset << PAGE_SHIFT));
            return 0;
        }
    } else {
        /*
         * Save the allocation parameters for use in nv_free_pages, where
         * we won't have access to all of them (sigh).
         */
        nv_alloc_private_t *ap;

        /* XXX Fix me! (PAT) */
        if (cache == NV_MEMORY_WRITECOMBINED)
            return -ENOMEM;

        ap = malloc(sizeof(nv_alloc_private_t), M_NVIDIA, M_WAITOK | M_ZERO);
        if (!ap) {
            return -ENOMEM;
        }
        *private = (void *) ap;

        ap->count = count;
        ap->alloc_type_contiguous = alloc_type_contiguous;
        ap->alloc_type_cached = alloc_type_cached;
        ap->alloc_type_kernel = alloc_type_kernel;
        ap->class = class;

        if (alloc_type_kernel &&
            !alloc_type_contiguous && alloc_type_cached) {
            /*
             * Obtain general system memory for the resource mananger, it
             * doesn't need to be contiguous.
             */
            if (nv_alloc_system_pages(nv, address, count))
                goto failed;

        } else if (alloc_type_contiguous ||
                   (alloc_type_kernel && !alloc_type_cached)) {
            /*
             * Make sure the pages are contiguous in physical memory; the
             * memory will not be used for DMA push buffers, the returned
             * linear mapping is expected to be usable as-is.
             */
            if (nv_alloc_contig_pages(nv, address, count))
                goto failed;

            /* XXX Fix me! (uc kernel) */
            if (alloc_type_kernel && !alloc_type_cached)
                /* goto failed */;

        } else if (!alloc_type_kernel) {
            /*
             * Allocate general system memory for DMA push buffers. These
             * pages may be non-contiguous; the individual addresses must
             * be retrievable from the mmap(2) implementation.
             */
            if (nv_alloc_system_pages(nv, address, count))
                goto failed;
            *address = (void *)NV_VM_TO_MMAP_OFFSET((NvUPtr)*address);
        }

        return 0;

failed:
        free(*private, M_NVIDIA);
    }

    return -ENOMEM;
}

S032 nv_free_pages(
    nv_state_t *nv,
    void **address,
    U032 count,
    U032 alloc_type_agp,
    void *private
)
{
    U032 offset;
    offset = ((U032)(NvUPtr)*address - nv->agp.address) >> PAGE_SHIFT;

    if (alloc_type_agp) {
        if (!NV_AGP_ENABLED(nv))
            return -EINVAL;

#ifdef NV_SUPPORT_OS_AGP
        if (NV_OSAGP_ENABLED(nv)) {
            if (nv_free_agp_pages(nv, address, count, private) != 0) {
                /*
                 * The following is a sanity check, this mustn't ever fail in
                 * in real-world scenario.
                 */
                os_dbg_breakpoint();
                return -EINVAL;
            }

            rm_free_agp_bitmap(nv, offset, count);
            return 0;
        }
#endif /* NV_SUPPORT_OS_AGP */

        if (NV_NVAGP_ENABLED(nv)) {
            nv_alloc_t *at;
            struct nvidia_softc *sc = nv->os_state;

            if (rm_free_agp_pages(nv, address, private) != RM_OK) {
                /*
                 * Don't free the underlying system memory unless the request
                 * to free the RM AGP resources completes successfully, which
                 * should never actually fail.
                 */
                os_dbg_breakpoint();
                return -EINVAL;
            }

            SLIST_FOREACH(at, &sc->alloc_list, list) {
                if (at->offset == offset) {
                    nv_free_vm_object(nv, (void *)at->object);
                    break;
                }
            }

            return 0;
        }
    } else {
        nv_alloc_private_t *ap;

        ap = (nv_alloc_private_t *) private;
        if (!ap) {
            os_dbg_breakpoint();
            return -EINVAL;
        }

        if (ap->alloc_type_kernel &&
            !ap->alloc_type_contiguous && ap->alloc_type_cached) {
            /*
             * Generic system memory used by the resource manager; almost
             * identical to the user memory case.
             */
            if (nv_free_system_pages(nv, *address))
                return -EINVAL;

        } else if (ap->alloc_type_contiguous ||
                   (ap->alloc_type_kernel && !ap->alloc_type_cached)) {
            /*
             * Free contigous system memory; this resource is rarely used
             * and hence managed globally.
             */
            if (nv_free_contig_pages(nv, *address))
                return -EINVAL;

            /* XXX Fix me! (uc kernel) */
            if (ap->alloc_type_kernel && !ap->alloc_type_cached)
                /* return -EINVAL */;

        } else if (!ap->alloc_type_kernel) {
            /*
             * XXX Generic system memory used for DMA push buffers. There
             * are munmap(2) considerations in this case on Linux, are we
             * really safe here?
             */
            *address = (void *)NV_MMAP_TO_VM_OFFSET((NvUPtr)*address);
            if (nv_free_system_pages(nv, *address))
                return -EINVAL;
        }

        free(ap, M_NVIDIA);
        return 0;
    }

    return -EINVAL;
}

ULONG nv_get_kern_phys_address(ULONG address)
{
    vm_offset_t va = (vm_offset_t) address;

#if defined(NVCPU_X86_64)
    if (va >= DMAP_MIN_ADDRESS && va < DMAP_MAX_ADDRESS)
        return DMAP_TO_PHYS(va);
#endif

    if (va < VM_MIN_KERNEL_ADDRESS) {
        os_dbg_breakpoint();
        return 0;
    }

    return vtophys(va);
}

ULONG nv_get_user_phys_address(ULONG address)
{
    struct vmspace *vm;
    vm_offset_t va = (vm_offset_t) address;

    if (va >= VM_MIN_KERNEL_ADDRESS) {
        os_dbg_breakpoint();
        return 0;
    }

    /* if (vm_fault_quick((caddr_t) va, VM_PROT_WRITE))
        return 0; */

    vm = curproc->p_vmspace;
    return pmap_extract(vmspace_pmap(vm), va);
}


int nvidia_mmap_dev(
    struct nvidia_softc *sc,
    vm_offset_t offset,
    vm_offset_t *physical
)
{
    nv_alloc_t *at;
    nv_state_t *nv = sc->nv_state;

    NV_PCI_CHECK_CONFIG_SPACE(nv);

    /*
     * Offsets that fall into the frame buffer, registry or AGP
     * apertures are physical addresses and mapped into userspace
     * directly.
     */
    if (IS_FB_OFFSET(nv, offset, PAGE_SIZE)) {
        *physical = offset;
        return 0;
    }

    if (IS_REG_OFFSET(nv, offset, PAGE_SIZE)) {
        *physical = offset;
        return 0;
    }

    if (IS_AGP_OFFSET(nv, offset, PAGE_SIZE)) {
        *physical = offset;
        return 0;
    }

    offset = NV_MMAP_TO_VM_OFFSET(offset);

    SLIST_FOREACH(at, &sc->alloc_list, list) {
        if (offset >= at->address &&
                offset < at->address + at->size) {
            *physical = vtophys(offset);
            return 0;
        }
    }

    return -1;
}


int nv_int10h_call(
    nv_state_t * nv,
    U032 *eax,
    U032 *ebx,
    U032 *ecx,
    U032 *edx,
    void *buffer
)
{
#if defined(NVCPU_X86) && defined(NV_USE_OS_VM86_INT10CALL)
    struct vm86frame vmf;

    vmf.vmf_eax = *eax;
    vmf.vmf_ebx = *ebx;
    vmf.vmf_ecx = *ecx;
    vmf.vmf_edx = *edx;

    if (vm86_intcall(0x10, &vmf) == 0) {
        *eax = vmf.vmf_eax;
        *ebx = vmf.vmf_ebx;
        *ecx = vmf.vmf_ecx;
        *edx = vmf.vmf_edx;
        return RM_OK;
    }
#endif /* NV_USE_OS_VM86_INT10CALL */

    return RM_ERROR;  /* XXX Fix me? (vm86_intcall) */
}

void nvidia_rc_timer(void *data)
{
    nv_state_t *nv = (nv_state_t *) data;
    struct nvidia_softc *sc;
    
    NV_PCI_CHECK_CONFIG_SPACE(nv);

    /*
     * We need this timer to trigger again one second from
     * now, reset the timeout.
     */
    rm_run_rc_callback(nv);

    sc = nv->os_state;
    sc->timer_ch = timeout(nvidia_rc_timer, (void *) nv, hz);
}

int nv_start_rc_timer(
    nv_state_t *nv
)
{
    struct nvidia_softc *sc = nv->os_state;

    if (nv->rc_timer_enabled != 0)
        return -EIO;

    sc->timer_ch = timeout(nvidia_rc_timer, (void *) nv, hz);
    nv->rc_timer_enabled = 1;

    return 0;
}

int nv_stop_rc_timer(
    nv_state_t *nv
)
{
    struct nvidia_softc *sc = nv->os_state;

    if (nv->rc_timer_enabled == 0)
        return -EIO;

    untimeout(nvidia_rc_timer, (void *) nv, sc->timer_ch);
    nv->rc_timer_enabled = 0;

    return 0;
}

void* nv_get_adapter_state(
    U016 bus,
    U016 slot
)
{
    unsigned int i;
    struct nvidia_softc *sc;
    nv_state_t *nv;

    for (i = 0; i < NV_MAX_DEVICES; i++) {
        sc = devclass_get_softc(nvidia_devclass, i);
        if (!sc)
            continue;
        nv = sc->nv_state;

        if (nv->bus == bus && nv->slot == slot)
            return (void *) nv;
    }

    return NULL;
}

void nv_verify_pci_config(nv_state_t *nv)
{
    NV_PCI_CHECK_CONFIG_SPACE(nv);
}

