/**
 * @file SimpleTextLogPrinterTest.h
 * @author Bartosz Janiak
 * 
 * @brief
 *
 * Copyright  2009 NVIDIA Corporation. All rights reserved.
 *
 * NVIDIA Corporation and its licensors retain all intellectual property and proprietary
 * rights in and to this software, related documentation and any modifications thereto. Any
 * use, reproduction, disclosure or distribution of this software and related documentation
 * without express license agreement from NVIDIA Corporation is strictly prohibited.
 */

#pragma once
#include "MockWinApi.h"
#include "cxxtest/TestSuite.h"
#include "TestObjects.h"
#include <Logging.h>
#include <SimpleTextLogPrinter.h>

#include <boost/assign/std/vector.hpp>
#include <boost/algorithm/string/erase.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/lexical_cast.hpp>

namespace Nvidia {
namespace Logging {
namespace Test {

using namespace boost::assign; // bring 'operator+=()' into scope

class SimpleTextLogPrinterTest : public CxxTest::TestSuite {
public:

    void TestComplex()
    {
        TestLoggable testLoggable;
        GenericSimpleLoggable<double> loggable(3.14);
        std::vector<int> vector;
        vector += 2, 5, 8;
        
        SimpleTextLogPrinter logPrinter;
        TestLogManager logManager;
        logPrinter.AddLogManager(&logManager);

        std::tstring scopeLoggerName(_T("scopeLoggerName"));

        long long timestampBefore = Utils::GetCurrentTimestamp() - Utils::GetFirstTimestamp();
        logPrinter.PrintMessage(LogInfo(LogLevel_Debug, scopeLoggerName, _T("function"), _T("file"), 123),
            MakeChainLoggable(MakeEmptyChain() + 42 + testLoggable + vector + loggable + _T("string")));
        long long timestampAfter = Utils::GetCurrentTimestamp() - Utils::GetFirstTimestamp();

        const std::tstring& message = logManager.GetMessages()[0];

        std::tstring expectedMessage;
        expectedMessage += _T(" |    DEBUG: [scopeLoggerName] 123@function : 42 ");
        expectedMessage += _T("TestLoggable123 vector<int> 3.1400000000000001 string. ");
        expectedMessage += _T("TestLoggable( id=\"123\" someField1=\"154\" someField2=TestLoggable2( id=\"abc\" doubleField=\"9.8699999999999992\" ) ");
        expectedMessage += _T("vectorField=vector( \"firstString\" \"secondString\" ) ) vector( \"2\" \"5\" \"8\" ) ");

#if defined UNICODE || defined _UNICODE
        boost::algorithm::replace_all(expectedMessage, _T("_CHAR_"), _T("wchar_t"));  
#else
        boost::algorithm::replace_all(expectedMessage, _T("_CHAR_"), _T("char"));
#endif
        
        TS_ASSERT_EQUALS(boost::algorithm::erase_head_copy(message, 11), expectedMessage);

        std::tstring timestampString = boost::algorithm::erase_tail_copy(message, (int)(message.size() - 11));
        boost::algorithm::trim(timestampString);
        long long timestamp = (long long)(boost::lexical_cast<double>(timestampString) * 1000.0);

        TS_ASSERT(timestampBefore <= timestamp);
        TS_ASSERT(timestamp <= timestampAfter);
    }
};

}
}
}

using namespace Nvidia::Logging::Test;