/**
 * @file LogInfo.h
 * @author Bartosz Janiak
 * 
 * @brief Contains the definition of LogInfo structure.
 *
 * Copyright  2009 NVIDIA Corporation. All rights reserved.
 *
 * NVIDIA Corporation and its licensors retain all intellectual property and proprietary
 * rights in and to this software, related documentation and any modifications thereto. Any
 * use, reproduction, disclosure or distribution of this software and related documentation
 * without express license agreement from NVIDIA Corporation is strictly prohibited.
 */

#pragma once

#include "LogLevel.h"
#include "ILogPrinter.h"
#include "Utils.h"

#include "TStd.h"
#include <boost/utility.hpp>

namespace Nvidia {
namespace Logging {

/// @brief Structure containing the additional information about the log message.
class LogInfo : private boost::noncopyable 
{
private:
    /// @brief Time in miliseconds from first call to GetTimestamp() to creation of this LogInfo instance.
    long long m_timestamp;
    LogLevel m_level;
    const std::tstring& m_scopeLoggerName;
    std::tstring m_functionName;
    std::tstring m_fileName;
    int m_lineNumber;

public:
    /// @brief Constructor. Sets values of all fields and generates the timestamp.
    LogInfo(LogLevel level, const std::tstring& scopeLoggerName, std::tstring functionName, std::tstring fileName, int lineNumber) :
        m_timestamp(Utils::GetCurrentTimestamp()),
        m_level(level),
        m_scopeLoggerName(scopeLoggerName),
        m_functionName(functionName),
        m_fileName(fileName),
        m_lineNumber(lineNumber) {}
    
    long long GetAbsoluteTimestamp() const
    {
        return m_timestamp;
    }

    long long GetRelativeTimestamp() const
    {
        return m_timestamp - Utils::GetFirstTimestamp();
    }

    LogLevel GetLevel() const
    {
        return m_level;
    }
    
    const std::tstring& GetScopeLoggerName() const
    {
        return m_scopeLoggerName;
    }

    const std::tstring& GetFunctionName() const
    {
        return m_functionName;
    }

    const std::tstring& GetFileName() const
    {
        return m_fileName;
    }

    int GetLineNumber() const
    {
        return m_lineNumber;
    }    

};

}
}