/**
 * @file Loggable.h
 * @author Bartosz Janiak
 * 
 * @brief Contains the definition of the Loggable and SimpleLoggable abstract classes.
 *
 * Copyright  2009 NVIDIA Corporation. All rights reserved.
 *
 * NVIDIA Corporation and its licensors retain all intellectual property and proprietary
 * rights in and to this software, related documentation and any modifications thereto. Any
 * use, reproduction, disclosure or distribution of this software and related documentation
 * without express license agreement from NVIDIA Corporation is strictly prohibited.
 */

#pragma once

#include "TStd.h"
#include "Utils.h"

#include <boost/lexical_cast.hpp>

namespace Nvidia {
namespace Logging {

/**
 * @brief Base class for objects that can output their string representation.
 *
 * Should not be used directly, please inherit from SimpleLoggable or CompositeLoggable instead.
 */
class Loggable
{
public:
    /// @brief Returns the string representation of this Loggable.
    virtual std::tstring GetTypeName() const;

    virtual std::tstring GetHumanReadableTypeName() const;

    /**
     * @brief Returns the string representation of the type of this Loggable.
     *
     * Preferalby this should be a short, human-readable representation of the type,
     * not necessarily reflecting the real type of the object.
     */
    virtual std::tstring GetStringValue() const = 0;
};

inline std::tstring Loggable::GetTypeName() const
{
    return boost::lexical_cast<std::tstring>(typeid(*this).name());
}

inline std::tstring Loggable::GetHumanReadableTypeName() const
{
    return Utils::GenerateHumanReadableTypeName(GetTypeName());

}

/**
* @brief Empty abstract class, used to identify Loggable%s that are not CompositeLoggable%s.
*/
class SimpleLoggable : public Loggable
{
};

}
}