/**
 * @file
 * @author Bartosz Janiak
 * 
 * @brief Contains tests for LoggableTraits.
 *
 * Copyright  2009 NVIDIA Corporation. All rights reserved.
 *
 * NVIDIA Corporation and its licensors retain all intellectual property and proprietary
 * rights in and to this software, related documentation and any modifications thereto. Any
 * use, reproduction, disclosure or distribution of this software and related documentation
 * without express license agreement from NVIDIA Corporation is strictly prohibited.
 */

#pragma once
#include "MockWinApi.h"
#include "cxxtest/TestSuite.h"
#include "TestObjects.h"
#include <Logging.h>

#include <boost/assign/std/vector.hpp>

namespace Nvidia {
namespace Logging {


using namespace boost::assign; // bring 'operator+=()' into scope

namespace Test {
    struct PodStruct 
    {
        int m_number;
        const char* m_string;
    };
}

template<>
class LoggableTraits<Test::PodStruct>
{
public:
    class PodStructLoggable : public CompositeLoggable
    {
    private:
        Test::PodStruct m_podStruct;

    public:
        PodStructLoggable(Test::PodStruct podStruct) : m_podStruct(podStruct) {}

        void LogContentsOn(LogContentsPrinter& logContentsPrinter) const
        {
            logContentsPrinter.LogContent(_T("number"), m_podStruct.m_number);
            logContentsPrinter.LogContent(_T("string"), m_podStruct.m_string);
        }

        std::tstring GetStringValue() const
        {
            return _T("PodStruct");
        }

        std::tstring GetHumanReadableTypeName() const
        {
            return _T("PodStruct");
        }
    };

    typedef PodStructLoggable LoggableType;
};

namespace Test {

class LoggableTraitsTest : public CxxTest::TestSuite
{
public:
    void TestLoggableTraits()
    {
        PodStruct podStruct = { 123, "abc" };

        SimpleTextLogPrinter logPrinter;
        TestLogManager logManager;
        logPrinter.AddLogManager(&logManager);

        std::tstring scopeLoggerName(_T("scopeLoggerName"));

        logPrinter.PrintMessage(LogInfo(LogLevel_Debug, scopeLoggerName, _T("function"), _T("file"), 123),
            MakeChainLoggable(MakeEmptyChain() + podStruct));

        const std::tstring& message = logManager.GetMessages()[0];

        std::tstring expectedMessage = _T(" |    DEBUG: [scopeLoggerName] 123@function : PodStruct. PodStruct( number=\"123\" string=\"abc\" ) ");

        TS_ASSERT_EQUALS(boost::algorithm::erase_head_copy(message, 11), expectedMessage);
    }
};

}
}
}

using namespace Nvidia::Logging::Test;